(*^
::[	Information =

	"This is a Mathematica Notebook file.  It contains ASCII text, and can be
	transferred by email, ftp, or other text-file transfer utility.  It should
	be read or edited using a copy of Mathematica or MathReader.  If you 
	received this as email, use your mail application or copy/paste to save 
	everything from the line containing (*^ down to the line containing ^*)
	into a plain text file.  On some systems you may have to give the file a 
	name ending with ".ma" to allow Mathematica to recognize it as a Notebook.
	The line below identifies what version of Mathematica created this file,
	but it can be opened using any other version as well.";

	FrontEndVersion = "Macintosh Mathematica Notebook Front End Version 2.2";

	MacintoshStandardFontEncoding; 
	
	fontset = title, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeTitle, center, M7, bold, e8,  24, "Times"; 
	fontset = subtitle, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeTitle, center, M7, bold, e6,  18, "Times"; 
	fontset = subsubtitle, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeTitle, center, M7, italic, e6,  14, "Times"; 
	fontset = section, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeSection, grayBox, M22, bold, a20,  18, "Times"; 
	fontset = subsection, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeSection, blackBox, M19, bold, a15,  14, "Times"; 
	fontset = subsubsection, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeSection, whiteBox, M18, bold, R65535, a12,  12, "Times"; 
	fontset = text, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = smalltext, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  10, "Times"; 
	fontset = input, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeInput, M42, N23, bold, L-5,  10, "Courier"; 
	fontset = output, output, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, L-5,  10, "Courier"; 
	fontset = message, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, R65535, L-5,  12, "Courier"; 
	fontset = print, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, L-5,  10, "Courier"; 
	fontset = info, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, B65535, L-5,  12, "Courier"; 
	fontset = postscript, PostScript, formatAsPostScript, output, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeGraphics, M7, l34, w245, h249,  12, "Courier"; 
	fontset = name, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7, italic,  10, "Geneva"; 
	fontset = header, inactive, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = leftheader, inactive, L2,  12, "Times"; 
	fontset = footer, inactive, noKeepOnOnePage, preserveAspect, center, M7,  12, "Times"; 
	fontset = leftfooter, inactive, L2,  12, "Times"; 
	fontset = help, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  10, "Times"; 
	fontset = clipboard, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = completions, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special1, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special2, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special3, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special4, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special5, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	paletteColors = 128; automaticGrouping; currentKernel; 
]
:[font = title; inactive; locked; preserveAspect; startGroup]
A Simulator for Feynman's Quantum Computer
:[font = subsubtitle; inactive; locked; preserveAspect]
Colin P. Williams
:[font = input; locked; initialization; preserveAspect]
*)
Off[General::spell1]
(*
:[font = section; inactive; locked; preserveAspect; startGroup]
Copyright Notice
:[font = text; inactive; locked; preserveAspect; endGroup]
Copyright Colin P. Williams (1997).

This Notebook is intended to be used in conjunction with "Explorations in Quantum Computing" by Colin P. Williams and Scott H. Clearwater, TELOS, Springer-Verlag (1997), ISBN:0-387-94768-X. Permission is hereby granted to copy and distribute this Notebook freely for any non-commercial activity provided you include this copyright notice at the beginning of all such copies. Please send suggestions and bug reports to Colin P. Williams at 
        colin@solstice.jpl.nasa.gov      (818) 306 6512 or 
        cpw@cs.stanford.edu               (415) 728 2118
For information on "Explorations in Quantum Computing" check out the TELOS web site:  http://www.telospub.com/catalog/PHYSICS/Explorations.html. To order call 1-800-777-4643.

All other rights reserved.
:[font = section; inactive; locked; preserveAspect; startGroup]
Overview
:[font = text; inactive; locked; preserveAspect]
This Notebook contains code for simulating Feynman's quantum computer (see "Explorations in Quantum Computing", Chapter 4).  It contains tools for building mathematical representations of quantum gates, tools for embedding quantum gates in quantum circuits and a simulator for the Feynman quantum computer that implements a given quantum circuit.   Be aware that classical simulations of quantum computers are very costly, computationally, so only simple circuits can be simulated within a reasonable time.  Nevertheless, the simulator is sufficient to illustrate several important features of quantum computation.
:[font = text; inactive; locked; preserveAspect]
You can build any classical circuit out of the following three types of reversible logic gates:
:[font = input; locked; preserveAspect]
NOTGate
CNGate
CCNGate
:[font = text; inactive; locked; preserveAspect]
These are the NOT gate, the CONTROLLED-NOT gate and the CONTROLLED-CONTROLLED-NOT gate respectively.  A particularly "quantum" gate is the one input/one output square root of NOT gate:
:[font = input; locked; preserveAspect]
SqrtNOTGate

:[font = text; inactive; locked; preserveAspect]
The SqrtNOTGate can be used to devise a circuit for computing the NOT function that is inherently "quantum" in the sense that it relies upon the Superposition Principle of quantum mechanics.  

Once you have designed your circuit,  you can simulate a Feynman quantum computer that implements this circuit using either of the commands:
:[font = input; locked; preserveAspect]
SchrodingerEvolution
EvolveQC
:[font = text; inactive; locked; preserveAspect]
SchrodingerEvolution evolves the quantum computer for a fixed period of time, EvolveQC evolves the quantum computer until the computation is completed.  In either case, you can visualize the resulting evolution using:
:[font = input; locked; preserveAspect]
PlotEvolution
:[font = text; inactive; locked; preserveAspect; endGroup]
See the Notebook ErrorsInQCs.ma to investigate the effects of errors in the operation of a quantum computer.  See the Notebook ErrorCorrection.ma to investigate techniques for quantum error correction.
:[font = section; inactive; preserveAspect; startGroup]
What Computation are we going to Simulate?
:[font = text; inactive; preserveAspect]
We are going to describe a quantum circuit that computes the NOT function via the application of two "square root of NOT" gates connected back to back.  There is no classical gate that can achieve the SqrtNOT operation i.e. there is no classical gate such that two consecutive applications of this (hypothetical, classical) gate yield the NOT operation.
:[font = subsection; inactive; preserveAspect; startGroup]
Gates & their Truth Tables
:[font = input; initialization; preserveAspect]
*)
NOT = {{0, 1},
       {1, 0}};
(*
:[font = input; initialization; preserveAspect]
*)
CN = {{1,0,0,0},
      {0,1,0,0},
      {0,0,0,1},
      {0,0,1,0}};
(*
:[font = input; initialization; preserveAspect; endGroup]
*)
CCN = {{1,0,0,0,0,0,0,0},
       {0,1,0,0,0,0,0,0},
       {0,0,1,0,0,0,0,0},
       {0,0,0,1,0,0,0,0},
       {0,0,0,0,1,0,0,0},
       {0,0,0,0,0,1,0,0},
       {0,0,0,0,0,0,0,1},
       {0,0,0,0,0,0,1,0}};
(*
:[font = subsection; inactive; preserveAspect; startGroup]
Creation and Annihilation Operators
:[font = input; initialization; preserveAspect]
*)
aOP = {{0,1},   (* annihilation operator on a single bit *)
       {0,0}};
       
annihilationOP[i_, m_]:=
	Apply[Direct,
	      ReplacePart[Table[IdentityMatrix[2], {m}], aOP, i]
	     ]
(*
:[font = input; initialization; preserveAspect; endGroup]
*)
cOP = {{0,0},     (* creation operator that acts on a single bit *)
       {1,0}};

creationOP[i_, m_]:=
	Apply[Direct,
	      ReplacePart[Table[IdentityMatrix[2], {m}], cOP, i]
	     ]
	
(*
:[font = subsection; inactive; preserveAspect; startGroup]
Rewriting Gates using Creation & Annihilation Operators
:[font = text; inactive; preserveAspect]
We can express the NOT, CONTROLLED-NOT and CONTROLLED-CONTROLLED-NOT gates as sums and products of creation and annihilation operators. Moreover, we can embed gates in a circuit that contains multiple input and output lines.  For example, NOTGate[i,m] is a NOT gate acting on the i-th of m inputs.  CNGate[i,j,m] is a CONTROLLED-NOT gate acting on the i-th and j-th of m inputs etc.
:[font = input; initialization; preserveAspect; endGroup]
*)
NOTGate[i_, m_]:=
	creationOP[i,m] + annihilationOP[i,m]

CNGate[i_, j_, m_]:=
	(creationOP[i,m] .
	 annihilationOP[i,m] . (annihilationOP[j,m] + creationOP[j,m]) +
	 annihilationOP[i,m] . creationOP[i,m]
	)
	
	
CCNGate[i_, j_, k_, m_]:=
	(IdentityMatrix[2^m] + 
	  creationOP[i,m] . 
	  annihilationOP[i,m] . 
	  creationOP[j,m] . 
	  annihilationOP[j,m] . 
	   (annihilationOP[k,m] + creationOP[k,m] - IdentityMatrix[2^m])
	)
	
(*
:[font = subsection; inactive; preserveAspect; startGroup]
Square Root of NOT Gate (SqrtNOT)
:[font = text; inactive; preserveAspect]
A SqrtNOT gate is a 1 input / 1 output quantum gate.
:[font = postscript; PostScript; formatAsPostScript; output; inactive; preserveAspect; pictureLeft = 34; pictureWidth = 248; pictureHeight = 82]
%!
%%Creator: Mathematica
%%AspectRatio: .33333 
MathPictureStart
%% Graphics
/Courier-Bold findfont 9  scalefont  setfont
% Scaling calculations
0.34127 0.31746 0.00793651 0.31746 [
[ 0 0 0 0 ]
[ 1 .33333 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
[ ] 0 setdash
0 g
p
P
0 0 m
1 0 L
1 .33333 L
0 .33333 L
closepath
clip
newpath
p
1 Mabswid
.02381 .16667 m
.34127 .16667 L
.34127 .00794 L
.65873 .16667 L
.34127 .3254 L
.34127 .00794 L
s
.65873 .16667 m
.97619 .16667 L
s
[(SqrtNOT)] .5 .16667 0 0 Mshowa
P
% End of Graphics
MathPictureEnd

:[font = subsubsection; inactive; Cclosed; preserveAspect; startGroup]
Code for sketching SqrtNOTGate
:[font = input; preserveAspect]
DrawSqrtNOTGate[x_]:=
	Graphics[{AbsoluteThickness[1],
	          Line[{{x-1,.5},{x,.5},{x,0},{1+x,.5},{x,1},{0+x,0}}],
	          Line[{{x+1,.5}, {x+2,.5}}],
	          Text["SqrtNOT", {x+.5,.5}, {0,0}]},
	             AspectRatio->Automatic];
	     
:[font = input; preserveAspect; startGroup]
Show[DrawSqrtNOTGate[0]]
:[font = postscript; PostScript; formatAsPostScript; output; inactive; preserveAspect; pictureLeft = 34; pictureWidth = 248; pictureHeight = 82]
%!
%%Creator: Mathematica
%%AspectRatio: .33333 
MathPictureStart
%% Graphics
/Courier-Bold findfont 9  scalefont  setfont
% Scaling calculations
0.34127 0.31746 0.00793651 0.31746 [
[ 0 0 0 0 ]
[ 1 .33333 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
[ ] 0 setdash
0 g
p
P
0 0 m
1 0 L
1 .33333 L
0 .33333 L
closepath
clip
newpath
p
1 Mabswid
.02381 .16667 m
.34127 .16667 L
.34127 .00794 L
.65873 .16667 L
.34127 .3254 L
.34127 .00794 L
s
.65873 .16667 m
.97619 .16667 L
s
[(SqrtNOT)] .5 .16667 0 0 Mshowa
P
% End of Graphics
MathPictureEnd

:[font = output; output; inactive; preserveAspect; endGroup]
Graphics["<<>>"]
;[o]
-Graphics-
:[font = input; preserveAspect]
DrawNOTCircuit[x_, y_]:=
	Show[DrawSqrtNOTGate[x], DrawSqrtNOTGate[y]]
	     
:[font = input; preserveAspect; startGroup]
DrawNOTCircuit[0,2]
:[font = postscript; PostScript; formatAsPostScript; output; inactive; preserveAspect; pictureLeft = 34; pictureWidth = 354; pictureHeight = 70]
%!
%%Creator: Mathematica
%%AspectRatio: .2 
MathPictureStart
%% Graphics
/Courier-Bold findfont 9  scalefont  setfont
% Scaling calculations
0.214286 0.190476 0.0047619 0.190476 [
[ 0 0 0 0 ]
[ 1 .2 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
[ ] 0 setdash
0 g
p
P
0 0 m
1 0 L
1 .2 L
0 .2 L
closepath
clip
newpath
p
p
1 Mabswid
.02381 .1 m
.21429 .1 L
.21429 .00476 L
.40476 .1 L
.21429 .19524 L
.21429 .00476 L
s
.40476 .1 m
.59524 .1 L
s
[(SqrtNOT)] .30952 .1 0 0 Mshowa
P
p
1 Mabswid
.40476 .1 m
.59524 .1 L
.59524 .00476 L
.78571 .1 L
.59524 .19524 L
.59524 .00476 L
s
.78571 .1 m
.97619 .1 L
s
[(SqrtNOT)] .69048 .1 0 0 Mshowa
P
P
% End of Graphics
MathPictureEnd

:[font = output; output; inactive; preserveAspect; endGroup; endGroup]
Graphics["<<>>"]
;[o]
-Graphics-
:[font = subsubsection; inactive; preserveAspect; startGroup]

:[font = text; inactive; preserveAspect]
A SqrtNOT gate that acts on the i-th of m qubits can be defined in terms of creation and annihilation operators as follows (in Mathematica I is the square root of -1):
;[s]
3:0,0;127,1;138,0;168,-1;
2:2,13,9,Times,0,12,0,0,0;1,13,9,Times,2,12,0,0,0;
:[font = input; initialization; preserveAspect]
*)
SqrtNOTGate[i_, m_]:=
	Module[{cOPim, aOPim},
		cOPim = creationOP[i,m];
		aOPim = annihilationOP[i,m];
		(1/2 (1 - I) (cOPim + aOPim) + 
	     1/2 (1+I) (aOPim . cOPim + cOPim . aOPim)
	    )
	]
(*
:[font = text; inactive; preserveAspect; endGroup]
It is interesting to examine the truth table for SqrtNOTGate. Unlike, any classical gate which always outputs bits (0s and 1s), the SqrtNOTGate can return superpositions of bits.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; startGroup]
TruthTable[SqrtNOTGate[1,1]]
:[font = output; output; inactive; preserveAspect; fontLeading = 0; endGroup]
ket[0] -> (1/2 + I/2)*ket[0] + (1/2 - I/2)*ket[1]
ket[1] -> (1/2 - I/2)*ket[0] + (1/2 + I/2)*ket[1]
;[o]
           1   I            1   I
ket[0] -> (- + -) ket[0] + (- - -) ket[1]
           2   2            2   2
           1   I            1   I
ket[1] -> (- - -) ket[0] + (- + -) ket[1]
           2   2            2   2
:[font = text; inactive; preserveAspect]
Why do we call this gate a SqrtNOTGate?  To answer this look at the truth table of two SqrtNOT gates connected back to back:
:[font = postscript; PostScript; formatAsPostScript; output; inactive; preserveAspect; pictureLeft = 34; pictureWidth = 354; pictureHeight = 70; endGroup]
%!
%%Creator: Mathematica
%%AspectRatio: .2 
MathPictureStart
%% Graphics
/Courier-Bold findfont 9  scalefont  setfont
% Scaling calculations
0.214286 0.190476 0.0047619 0.190476 [
[ 0 0 0 0 ]
[ 1 .2 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
[ ] 0 setdash
0 g
p
P
0 0 m
1 0 L
1 .2 L
0 .2 L
closepath
clip
newpath
p
p
1 Mabswid
.02381 .1 m
.21429 .1 L
.21429 .00476 L
.40476 .1 L
.21429 .19524 L
.21429 .00476 L
s
.40476 .1 m
.59524 .1 L
s
[(SqrtNOT)] .30952 .1 0 0 Mshowa
P
p
1 Mabswid
.40476 .1 m
.59524 .1 L
.59524 .00476 L
.78571 .1 L
.59524 .19524 L
.59524 .00476 L
s
.78571 .1 m
.97619 .1 L
s
[(SqrtNOT)] .69048 .1 0 0 Mshowa
P
P
% End of Graphics
MathPictureEnd

:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect]
TruthTable[SqrtNOTGate[1,1] . SqrtNOTGate[1,1]]
:[font = text; inactive; preserveAspect; endGroup]
The net result of two SqrtNOTGates connected back to back, is a unitary operation (in fact the NOT operation). 
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect]
UnitaryQ[SqrtNOTGate[1,1] . SqrtNOTGate[1,1]]
:[font = input; preserveAspect; endGroup; endGroup; endGroup]
UnitaryQ[NOT]
:[font = section; inactive; preserveAspect; startGroup]
Representing the Computation as a Circuit
:[font = text; inactive; preserveAspect]
In general , the quantum memory register of a Feynman-like quantum computer will consist of a set of "cursor" qubits (which keep track of the progress of the computation) and a set of "program" qubits through which the "input" is fed into the computer and the "output" is extracted (when it becomes available). We want to apply specific operators, corresponding to the action of logic gates, on just the "program" qubits.  Hence we must specify which qubits a given operator must act upon.  In the case of the simple SqrtNOTGate squared circuit, we will use two SqrtNOTGates that both act on the 4th of 4 qubits.  Thus the NOT circuit, built from two SqrtNOTGates connected back to back, is specified as follows:
:[font = input; initialization; preserveAspect]
*)
sqrtNOTcircuit = {SqrtNOTGate[4,4], SqrtNOTGate[4,4]};
(*
:[font = text; inactive; preserveAspect]
This contains an ordered list of quantum gates together with the input "lines" these gates work upon.

The embedded NOT gate (NOTGate[i,m]), the embedded CONTROLLED-NOT gate (CNGate[i,j,m]) and the embedded CONTROLLED-CONTROLLED-NOT gate (CCN[i,j,k,m]) are defined similarly. The arguments i, j, k label the lines of the gates and m signifies the total number of lines in the circuit. Thus, a controlled-NOT gate acting on the 2nd and 4th of 5 lines would be CNGate[2,4,5] etc.

You can compare the square root of NOT gate and its embedded form easily.  A square root of NOT gate acting on the 1st of 1 qubit (i.e. an unembedded SqrtNOTGate) is given by sqrtNOT11 in the example below:
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect]
SetOptions[$Output, PageWidth->Infinity];

sqrtNOT11 = SqrtNOTGate[1,1];  (* unembedded gate *)

MatrixForm[sqrtNOT11, 
           TableSpacing->{0,4}, TableAlignments->{Center, Center}]
:[font = text; inactive; preserveAspect; endGroup]
The square root of NOT gate that acts on the 2nd of 3 qubits is given by sqrtNOT23 in the next example:
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect]
SetOptions[$Output, PageWidth->Infinity];

sqrtNOT23 = SqrtNOTGate[2,3];

MatrixForm[sqrtNOT23, 
           TableSpacing->{0,4}, TableAlignments->{Center, Center}]
:[font = text; inactive; preserveAspect; endGroup]
Other embedded gates may be derived in a similar fashion.

Notice that the embedded gates are still unitary:
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; endGroup; endGroup]
UnitaryQ[sqrtNOT23]
:[font = section; inactive; preserveAspect; startGroup]
Determining the Size of the Memory Register
:[font = text; inactive; preserveAspect; endGroup]
For the NOT circuit, built from two SqrtNOTGates connected back to back, there are 2 gates (k=2), so we need need k+1=3 cursor qubits (to track the progress of the computation) and 1 input/output qubit (m=1) which will serve a dual purpose; to enter the input to the circuit and to record the output. Hence we need 4 qubits in all; 3 cursor qubits plus one program qubit, making a total of m+k+1=4 qubits for the entire quantum memory register.
:[font = section; inactive; preserveAspect; startGroup]
Computing the Hamiltonian Operator
:[font = text; inactive; preserveAspect]
In Feynman's quantum computer, the Hamiltonian is time independent and consists of a sum of terms describing the advance and retreat of the computation.  The net effect of the Hamiltonian is to place the memory register of the Feynman quantum computer in a superposition of states representing the same computation at various stages of completion. The command for generating the Hamiltonian of a Feynman quantum computer is "Hamiltonian".
:[font = input; preserveAspect; startGroup]
?Hamiltonian
:[font = print; inactive; preserveAspect; endGroup]
Hamiltonian[m, k, circuit] returns the time independent Hamiltonian matrix
   corresponding to the given circuit. The circuit consists of k quantum gates
   and a total of m lines (i.e. m inputs and m outputs).
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; endGroup]
SetOptions[$Output, PageWidth->Infinity];
H = Hamiltonian[1, 2, sqrtNOTcircuit];
MatrixForm[H, TableSpacing->{0,4}, TableAlignments->{Center,Center}]
:[font = subsubsection; inactive; Cclosed; preserveAspect; startGroup]
Code
:[font = input; initialization; preserveAspect; startGroup]
*)
Hamiltonian[m_, k_, circuit_]:=
	Module[{terms},
		terms = Table[term[i, m+k+1, circuit[[i]]], {i, 1, k}];
		Apply[Plus, terms + Map[Conjugate[Transpose[#]]&, terms]]
	]

Hamiltonian::usage =
	"Hamiltonian[m, k, circuit] returns the time independent \
	Hamiltonian matrix corresponding to the given circuit. \
	The circuit consists of k quantum gates and a total of \
	m lines (i.e. m inputs and m outputs).";
	
term[i_, l_, gate_]:=
	creationOP[i+1, l] . annihilationOP[i, l] . gate
(*
:[font = message; inactive; preserveAspect; endGroup; endGroup; endGroup]
General::spell1: 
   Possible spelling error: new symbol name "term" is similar to existing symbol 
    "terms".
:[font = section; inactive; preserveAspect; startGroup]
Computing the Unitary Evolution Operator
:[font = text; inactive; preserveAspect]
The unitary evolution operator is derived from the solution to Schrodinger's equation for the evolution of the memory register i.e. | psi(t) > = U(t) | psi(0) > where U(t) = e^(-i H t / hBar) where H is the (time-independent) Hamiltonian, i is square root of -1, hBar is Planck's constant over 2 Pi and t is the time. Remember, U(t) and H are really square matrices so the necessary exponential is a matrix exponential.  We set hBar=1 for simplicity.   In terms of the code, we call U(t), for a particular circuit at a particular time, EvolutionOP[m,k,circuit,t].
:[font = input; preserveAspect; startGroup]
?EvolutionOP
:[font = print; inactive; preserveAspect; endGroup]
EvolutionOP[m, k, circuit, t] is the time dependent evolution operator for the
   given circuit. The circuit consists of k quantum gates and a total of m lines
   (i.e. m inputs and m outputs).
:[font = text; inactive; preserveAspect]
U is given by the t-th matrix power of expH i.e. U = MatrixPower[expH, t] where expH is the matrix exponential of H.  You may need to scroll sideways to see the structure of expH.
:[font = subsubsection; inactive; Cclosed; preserveAspect; startGroup]
Code
:[font = input; initialization; preserveAspect; endGroup]
*)
EvolutionOP[m_, k_, circuit_, t_ ]:=
	MatrixPower[N[MatrixExp[ -I Hamiltonian[m, k, circuit]]], t]
	
EvolutionOP::usage =
	"EvolutionOP[m, k, circuit, t] is the time dependent evolution \
	operator for the given circuit. The circuit consists of k \
	quantum gates and a total of m lines (i.e. m inputs and m \
	outputs).";
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; endGroup; endGroup]
SetOptions[$Output, PageWidth->Infinity];
expH = MatrixExp[ -I Hamiltonian[1, 2, sqrtNOTcircuit]];
MatrixForm[expH, TableSpacing->{0,0}]
:[font = section; inactive; preserveAspect; startGroup]
Running the Quantum Computer for a Fixed Length of Time
:[font = text; inactive; preserveAspect]
To simulate running the quantum computer, that implements a given circuit, for a specific length of time, use the function SchrodingerEvolution.  This function takes 5 arguments: the initial state of the memory register, the number of inputs/outputs in the circuit, the number of gates in the circuit, the circuit matrix expressed as a product of embedded gates and the duration of the simulation.  The result is the state of the memory register at the end of the simulation period.
:[font = input; preserveAspect; startGroup]
?SchrodingerEvolution
:[font = print; inactive; preserveAspect; endGroup]
SchrodingerEvolution[initKet, m, k, circuit, t] evolves the given circuit for
   time t from the initial configuration initKet (a ket vector e.g.
   ket[1,0,0,0]). You need to specify that there are k gates in the circuit
   (so there are k+1 cursor bits) and that there are m program bits (i.e. the
   number of bits used as input data not counting the cursor bits).
:[font = subsubsection; inactive; Cclosed; preserveAspect; startGroup]
Code
:[font = input; initialization; preserveAspect; endGroup]
*)
SchrodingerEvolution[ket_, m_, k_, circuit_, t_]:=
	ColumnVectorToKet[Chop[EvolutionOP[m,k,circuit,t] . 
	                       KetToColumnVector[ ket ]
	                      ]
	                 ]
SchrodingerEvolution::usage =
	"SchrodingerEvolution[initKet, m, k, circuit, t] \
	evolves the given circuit for time t from the initial configuration \
	initKet (a ket vector e.g. ket[1,0,0,0]). You need to specify that \
	there are k gates in the circuit (so there are k+1 cursor bits) and \
	that there are m program bits (i.e. the number of bits used as input \
	data not counting the cursor bits).";
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; startGroup]
sqrtNOTcircuit = {SqrtNOTGate[4,4], SqrtNOTGate[4,4]};
SchrodingerEvolution[ket[1,0,0,0], 1, 2, sqrtNOTcircuit, 0.5]
:[font = output; output; inactive; preserveAspect; endGroup]
-0.1198777014621846*ket[0, 0, 1, 1] + 
 
  (0.229681342466392 - 0.2296813424663922*I)*ket[0, 1, 0, 0] + 
 
  (-0.2296813424663921 - 0.2296813424663922*I)*ket[0, 1, 0, 1] + 
 
  0.880122298537815*ket[1, 0, 0, 0]
;[o]
-0.119878 ket[0, 0, 1, 1] + (0.229681 - 0.229681 I) ket[0, 1, 0, 0] + 
 
  (-0.229681 - 0.229681 I) ket[0, 1, 0, 1] + 0.880122 ket[1, 0, 0, 0]
:[font = text; inactive; preserveAspect; endGroup]
Is this a properly normalized ket?
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; startGroup]
NormalizedKetQ[%]
:[font = output; output; inactive; preserveAspect; endGroup]
True
;[o]
True
:[font = text; inactive; preserveAspect; endGroup]
Do you get the same answer is you run the simulator again, on the same input, for the same length of time?
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect]
sqrtNOTcircuit = {SqrtNOTGate[4,4], SqrtNOTGate[4,4]};
SchrodingerEvolution[ket[1,0,0,0], 1, 2, sqrtNOTcircuit, 0.5]
:[font = text; inactive; preserveAspect; endGroup]
Were you surprised by the outcome?
:[font = subsubsection; inactive; Cclosed; preserveAspect; startGroup]
Explanation in here ...
:[font = text; inactive; preserveAspect; endGroup; endGroup]
The Schrodinger equation is completely deterministic. Thus if you evolve the same circuit, with the same input, for the same length of time, the outcome, ignoring errors, will be identical each time. It is only when you make a measurement that randomness enters the picture.
        Be aware that some physicists would contest the last statement.  Outcomes of measurements are random according to the "Copenhagen" interpretation of quantum mechanics.  Other interpretations, such as Everitt's "Many-Worlds" interpretation and Cerf & Adami's interpretation, explain measurement differently.  However, all interpretations accept that the Schrodinger equation is a deterministic differential equation.
:[font = section; inactive; preserveAspect; startGroup]
Running the Quantum Computer Until the Computation is Done
:[font = text; inactive; preserveAspect]
In a Feynman-like quantum computer, the position of the cursor keeps track of the logical progress of the computation. If the computation can be accomplished in k+1 (logic gate) operations, the cursor will consist of a chain of k+1 atoms, only one of which can ever be in the |1> state. The cursor keeps track of how many,	logical operations have been applied to the program bits thus far. Thus if you measure the cursor position and find it at the third site, say, then you know that the memory register will, at that moment, contain the result of applying the first three gate operations to the input state. This does not mean that only three such operations have been applied. In the Feynman computer the computation proceeds forwards and backwards simultaneously. As time progresses, the probability of finding the cursor at the (k+1)-th site rises and falls. If you are lucky, and happen to measure the cursor position when the probability of the cursor being at the (k+1)-th site is high, then you have a good chance of finding it there.

Operationally, you periodically measure the cursor position. This collapses the superposition of states that represent the cursor position but leaves the superposition of states in the program bits unscathed. If the cursor is not at the (k+1)-th site then you allow the computer to evolve again from the new, (partially) collapsed state. However, as soon as the cursor is found at the (k+1)-th site, the computation is halted and the complete state of the memory register (cursor bits and program bits) is measured. Whenever the cursor is at the (k+1)-th site, a measurement of the state of the program bits at that moment is guaranteed to return a valid answer to the computation the quantum computer was working on. So in the Feynman model of a quantum computer, there is no doubt at to the correctness of the answer, merely the time at which the answer is available.
:[font = text; inactive; preserveAspect]
To run the quantum computer until completion, you must periodically check the cursor position to see if all the gate operations have been applied. As soon as you find the cursor in its extreme position, you can be sure that, at that moment, a correct answer is obtainable from reading the program qubits. Note that we say "a" correct answer and not "the" correct answer because, if a problem admits more than one acceptable solution, then the final state of the  Feynman's quantum computer will contain a superposition of all the valid answers.  Upon measurement only one of these answers will be obtained however.

To read just the cursor, one simply restricts measurements of the memory register to be made on just those qubits used to keep track of the cursor.  Thus the program qubits (that contain the answer) are NOT measured in this process.  Nevertheless, the outcome of the measurement of the cursor position causes the relative state of the program qubits to be projected into a subspace that is consistent with the position of the cursor that is found.   For example, if the cursor position indicates that, say, the first N gate operations have been applied, then the program qubits are projected into a superposition corresponding to the state created by applying just the first N gate operations in the circuit.  This idea of measurements of one part of a memory register affecting the relative state of the other (unmeasured) part of the same register, is crucial to understanding the operation of quantum computers.  In general you do not want to make any measurements on the program qubits (the qubits which contain the answer) until you can be sure that an answer is available. The command for reading the cursor position is ReadCursorBits:
:[font = input; preserveAspect; startGroup]
?ReadCursorBits
:[font = print; inactive; preserveAspect; endGroup]
ReadCursorBits[numCursorBits, superposition] reads the state of the cursor
   bits of a Feynman-like quantum computer. If the computation can be
   accomplished in k+1 (logic gate) operations, the cursor will consist of a
   chain of k+1 atoms, only one of which can ever be in the |1> state. The
   cursor keeps track of how many, logical operations have been applied to
   the program bits thus far. The state of the program bits of the computer
   are unaffected by measuring just the cursor bits. If the cursor is ever
   found at the (k+1)-th site, then, if you measured the program bits at
   that moment, they would be guaranteed to contain a valid answer to the
   computation the quantum computer was working on.
:[font = text; inactive; preserveAspect]
For our sqrtNOTcircuit, we have 2 gates (i.e. k=2) and therefore k+1=3 cursor positions.  The initial state of the cursor is |100> and the input to the circuit is set to |0>.  Thus the overall memory register is initially in the state |1000> (which we represent as ket[1,0,0,0]).  To read the cursor position after a time t=1.1 has elapsed use ReadCursorBits[3, evoln1] where evoln1 is the state of the entire memory register after the computer has evolved for time t=1.1.
:[font = subsubsection; inactive; Cclosed; preserveAspect; startGroup]
Code
:[font = input; initialization; preserveAspect; endGroup]
*)
Options[ReadCursorBits] = {TraceProgress->False};

ReadCursorBits[numBitsToRead_, w_. ket[bits__] + kets_., opts___]:=
	Module[{nBits, superposition, resultsPerStep, traceQ},
		traceQ = TraceProgress /. {opts} /. Options[ReadCursorBits];
		nBits = Length[{bits}]; (* figure out number of bits in memory *)
		superposition = {"BeforeAnyMeasurements", w ket[bits] + kets};
		resultsPerStep = FoldList[MeasureIthBit[#2,#1,nBits]&,
		                          superposition,
		                          Range[numBitsToRead]
		                         ];
		Which[traceQ===False,
		        {Rest[Map[#[[1]]&, resultsPerStep]],  (* results for bits *)
		         Last[resultsPerStep][[2]]            (* projected state *)
		        },
		      traceQ===True,
		        ColumnForm[resultsPerStep]
		     ]
	]/;(numBitsToRead <= Length[{bits}])

ReadCursorBits::usage =
	"ReadCursorBits[numCursorBits, superposition] reads the state \
	of the cursor bits of a Feynman-like quantum computer. If the \
	computation can be accomplished in k+1 (logic gate) operations, \
	the cursor will consist of a chain of k+1 atoms, only one of which \
	can ever be in the |1> state. The cursor keeps track of how many, \
	logical operations have been applied to the program bits thus far. \
	The state of the program bits of the computer are unaffected by \
	measuring just the cursor bits. If the cursor is ever found at the \
	(k+1)-th site, then, if you measured the program bits at that moment, \
	they would be guaranteed to contain a valid answer to the computation \
	the quantum computer was working on.";
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect]
sqrtNOTcircuit = {SqrtNOTGate[4,4], SqrtNOTGate[4,4]};
state1  = SchrodingerEvolution[ket[1,0,0,0],1,2,sqrtNOTcircuit,1.1];

{cursor1, projected1} = ReadCursorBits[3, state1]
:[font = text; inactive; preserveAspect]
The output consists of a list, {cursor-position, projected-state}, showing the cursor position and the state of the entire memory register after the cursor position has been measured.  For example, the output:
:[font = output; output; inactive; preserveAspect]
{{0, 1, 0}, (0.5 - 0.5000000000000002*I)*ket[0, 1, 0, 0] + 
 
   (-0.5 - 0.5000000000000001*I)*ket[0, 1, 0, 1]}
;[o]
{{0, 1, 0}, (0.5 - 0.5 I) ket[0, 1, 0, 0] + (-0.5 - 0.5 I) ket[0, 1, 0, 1]}
:[font = text; inactive; preserveAspect; endGroup]
would indicate that the cursor is in the second position ({0,1,0}), and the state of the entire memory register is then  (0.5 - 0.5 i) |0100> + (-0.5 - 0.5 i) |0101> (where i is the square root of -1).  You can compare the state of the register after measuring the cursor position (projected1) to its state before measuring the cursor position (state1) by simply asking for the values of these variables.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
Compare the following two states:
:[font = input; preserveAspect]
projected1 
:[font = input; preserveAspect]
state1
:[font = text; inactive; preserveAspect; endGroup]
If you measure the cursor position of two identically prepared quantum computers at identical times would you find both cursors at the same place?
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
The initial set up is identical, is the result of reading the cursor?
:[font = input; preserveAspect]
sqrtNOTcircuit = {SqrtNOTGate[4,4], SqrtNOTGate[4,4]};
state2 = SchrodingerEvolution[ket[1,0,0,0],1,2,sqrtNOTcircuit,1.1];

{cursor2, projected2} = ReadCursorBits[3, state2]
:[font = text; inactive; preserveAspect]
Are the cursors found at the same position?
:[font = input; preserveAspect]
cursor1 === cursor2
:[font = text; inactive; preserveAspect]
Are the states of the memory registers the same after measuring the cursors?
:[font = input; preserveAspect]
projected1 === projected2
:[font = text; inactive; preserveAspect]
Were the states of the memory registers the same before the cursor positions were measured?
:[font = input; preserveAspect; endGroup]
state1 === state2
:[font = subsubsection; inactive; preserveAspect; startGroup]
Explanation in here ...
:[font = text; inactive; preserveAspect; endGroup]
In general, you will obtain different results for the cursor position in the two cases, although by chance you might have obtained the same answer both times!  If so, try running the last experiment again a few times.

If two identically prepared quantum computers are allowed to evolve for identical times, then, as the Schrodinger equation is deterministic, the two memory registers will evolve into identical superpositions  (ignoring errors of course).  Hence state1 will always be identical to state2.  However, when you measure the cursor positions, you cause each of the superpositions to "collapse" in a random way independent of one another.  In one case you may find, say, 2 gate operations have been applied, and in the other you may find say 1 gate operation has been applied.  Thus cursor1 is not, in general, the same as cursor2.   Consequently,  the relative states of the program qubits of each computer will then also be different after the measurements of the cursor.  Hence projected1 is different from projected2, in general.
:[font = subsection; inactive; preserveAspect; startGroup]
Reading the Memory Register (Cursor Qubits & Program Qubits)
:[font = text; inactive; preserveAspect]
Once the cursor is found at a position indicating that all of the gate operations have been applied, i.e. at its k+1-th site, then you can extract an answer from the computer by reading all the qubits in the memory register.
:[font = input; preserveAspect; startGroup]
?ReadMemoryRegister
:[font = print; inactive; preserveAspect; endGroup]
ReadMemoryRegister[superposition] reads the state of each bit in the memory
   register. As the i-th and j-th bit measurement operators commute (for any i
   and j), it does not matter in what order you measure the bits.
:[font = text; inactive; preserveAspect]
If a computation has only one answer, you will always obtain the single correct answer.  If a computation admits many possible answers, you are equally likely to obtain any one of them upon measuring the final state of the register.
:[font = subsubsection; inactive; Cclosed; preserveAspect; startGroup]
Code
:[font = input; initialization; preserveAspect]
*)
(*====================*)
(* ReadMemoryRegister *)
(*====================*)
(* Given a superposition representing the state of the memory of a
   quantum computer, return the result of measuring the memory.
*)
Options[ReadMemoryRegister] = {TraceProgress->False};

ReadMemoryRegister[w_. ket[bits__] + kets_., opts___]:=
	Module[{nBits, superposition, resultsPerStep, traceQ},
		traceQ = TraceProgress /. {opts} /. Options[ReadMemoryRegister];
		nBits = Length[{bits}]; (* figure out number of bits in memory *)
		superposition = {"BeforeAnyMeasurements", w ket[bits] + kets};
		resultsPerStep = FoldList[MeasureIthBit[#2,#1,nBits]&,
		                          superposition,
		                          Range[nBits]
		                         ];
		Which[traceQ===False, 
		        {Rest[Map[#[[1]]&, resultsPerStep]],  (* results for bits *)
		         Last[resultsPerStep][[2]]            (* projected state *)
		        },
		      traceQ===True,
		        ColumnForm[resultsPerStep] (*list of {results,projectedStates}*)
		     ]
	]
	
ReadMemoryRegister::usage =
	"ReadMemoryRegister[superposition] reads the state of each bit \
	in the memory register. As the i-th and j-th bit measurement \
	operators commute (for any i and j), it does not matter in what \
	order you measure the bits.";
	
(*
:[font = input; initialization; preserveAspect]
*)
MeasureIthBit[i_, {_, superposition_}, nBits_]:=
	Module[{p1, zeroOrOne, projectedState},
		p1 = ProbabilityIthBitIs1[i, superposition];
		zeroOrOne = BiasedSelect[{0,1}, {1-p1, p1}];
		projectedState = 
		  SuperpositionWithIthBitFixed[i, zeroOrOne, nBits, superposition];
		{zeroOrOne, projectedState}
	]
		       
KetWithIthBitZeroOrOne[i_, zeroOrOne_, nBits_]:=
	ReplacePart[Apply[ket, Table[_,{nBits}]], zeroOrOne, i]

SuperpositionWithIthBitFixed[_, _, _, w_. ket[bits__]]:=
  NormalizeKet[w ket[bits]]
SuperpositionWithIthBitFixed[i_, zeroOrOne_, nBits_, superposition_]:=
  NormalizeKet[Select[superposition,
		              MatchQ[#, _. KetWithIthBitZeroOrOne[i,zeroOrOne,nBits]
		                    ]&
		             ]
		      ]
(*
:[font = input; initialization; preserveAspect]
*)
ProbabilityIthBitIs1[i_, w_. ket[bits__]]:=
	If[ket[bits][[i]] == 1, Abs[w]^2, 1-Abs[w]^2]  (* Abs[w]^2 == 1 or 0 only *)
	
ProbabilityIthBitIs1[i_, w_. ket[bits__] + kets_.]:=
	Module[{nBits, terms},
		nBits = Length[{bits}];
		terms = Select[w ket[bits] + kets, 
		               MatchQ[#, _. KetWithIthBit1[i,nBits]]&];
		N[Apply[Plus, Map[Abs[#]^2&, Amplitudes[terms]]]/
		  Apply[Plus, Map[Abs[#]^2&, Amplitudes[w ket[bits] + kets]]]
		 ]
	]
	
ProbabilityIthBitIs1[i_, c_. (w_. ket[bits__] + kets_.)]:=
	ProbabilityIthBitIs1[i, Expand[c (w ket[bits] + kets)]]

ProbabilityIthBitIs1::usage =
	"The state of the memory register of a quantum computer \
	(that is comprised of m 2-state particles) is represented by a \
	superposition 2^m eigenstates. The function \
	ProbabilityIthBitIs1[i, superposition] computes the probability \
	that, upon being measured, the i-th, of the m, bits will be a 1.";
	
KetWithIthBit1[i_, nBits_]:=
	ReplacePart[Apply[ket, Table[_,{nBits}]], 1, i]
(*
:[font = input; initialization; preserveAspect; endGroup; endGroup]
*)
(* The list of probabilities should sum to 1. The call to Partition 
   constructs a set of probability intervals whose width is proportional
   to the probability with which the corresponding element in list 
   is selected.
*)
BiasedSelect[list_, probabilities_]:=
	Module[{random},
		random  = Random[];
		Apply[Part[list, #]&,
		      Flatten[
		       Position[Map[InRangeQ[random, #]&, 
		                    Partition[FoldList[Plus,0,probabilities],2,1]
		                   ],
		                True
		       ]
		      ]
		     ]
	] (* /;CheckProbabilitiesQ[probabilities] *)

BiasedSelect::usage =
	"BiasedSelect[{e1,e2,...,en}, {p1,p2,...,pn}] returns element ei of \
	the first list with probability given in the second list pi.";

BiasedSelect::probabilityLeak =
	"You have a probability leak. The probabilities you specified do \
	not add up to 1.";

BiasedSelect::excess =
	"The probabilities you specified sum to greater than 1.";

CheckProbabilitiesQ[probabilities_]:=
	Module[{psum = Apply[Plus, probabilities]},
		Which[psum<1,  Message[BiasedSelect::probabilityLeak],
		      psum>1,  Message[BiasedSelect::excess],
		      psum==1, True
		     ]
	]
	
InRangeQ[n_, {lb_, 1}]:=   lb <= n <= 1
InRangeQ[n_, {lb_, ub_}]:= lb <= n < ub
(*
:[font = subsection; inactive; preserveAspect; startGroup]
Evolving the Quantum Computer Until Complete
:[font = text; inactive; preserveAspect]
The above steps are bundled together in the command EvolveQC.  EvolveQC allows you to evolve the quantum computer, checking whether an answer is available and, if so,  measuring the memory register to extract it.  The output from EvolveQC is a complete history of the evolution of the quantum computer.  This history consists of a sequence of 4-element snapshots.  Each snapshot shows the time at which the cursor was measured, the state of the register before the cursor was measured, the result of the measurment and the state into which the register is projected because of the measurement.  You can use the option TimeBetweenObservations to control the time between observations of the cursor and whether the intervals should be regular or random. The default interval is 1 time unit.
:[font = input; preserveAspect; startGroup]
?EvolveQC
:[font = print; inactive; preserveAspect; endGroup]
The function EvolveQC[initState, circuit] evolves a Feynman-like quantum
   computer, specified as a circuit of interconnected quantum logic gates, from
   some initial state until the computation is complete. The output is a list of
   snapshots of the state of the QC at successive cursor-measurement times. Each
   snapshot consists of a 4 element list whose elements are the time at which the
   cursor is measured, the state of the register immediately before the cursor is
   measured, the result of the measurement and the state of the register
   immediately after the cursor position is measured. The latter is the projected
   state of the register. EvolveQC can take the optional argument
   TimeBetweenObservations which can be set to a number or a probability
   distribution. The default time between observations is 1 time unit.
:[font = subsubsection; inactive; Cclosed; preserveAspect; startGroup]
Code
:[font = input; initialization; preserveAspect; startGroup]
*)
Needs["Statistics`ContinuousDistributions`"]  (* for distributions *)
Needs["Statistics`DiscreteDistributions`"]

Options[EvolveQC] = {TimeBetweenObservations->1, Explain->False};
                    
EvolveQC[w_. ket[bits__] + kets_., circuit_, opts___]:=
	Module[{m, k, cursor, tau, explain, state, history},
		k = Length[circuit];   (* the circuit consists of k operators *)
		m = Length[{bits}]-(k+1);
		cursor = Join[{1}, Table[0,{k}]]; (* initial state of cursor *)
		tau = TimeBetweenObservations /. {opts} /. Options[EvolveQC];
		explain = Explain /. {opts} /. Options[EvolveQC];
		
		state = {0, w ket[bits] + kets, cursor, w ket[bits] + kets};
		history = {state};
		
		If[explain===True, ExplainResult[state]];
		While[Not[ComputationCompleteQ[state]] && CursorOkQ[state],
			      (state = EvolveForTimeTau[state,m,k,N[circuit],EvolnTime[tau]];
			       If[explain===True, ExplainResult[state]];
			       AppendTo[history, state]
			      )
			     ];
		
		Which[Not[CursorOkQ[state]],  
		              (Message[EvolveQC::badCursor]; 
		               AppendTo[history, 
		                        ReplacePart[state, $BadCursor, 3]
		                       ]
		              ), (* return history with bad cursor flag *)
		              
		       explain===False,       history,    (* return complete history *)
		       explain===True,        Print["You can now read an answer from the memory register that is guaranteed to be correct!"]
		     ]
	]
	
EvolveQC::badCursor = 
	"The cursor was corrupted. No useful answers may be extracted \
	from the computer.";

EvolveQC::usage =
	"The function EvolveQC[initState, circuit] evolves a Feynman-like \
	quantum computer, specified as a circuit of interconnected \
	quantum logic gates, from some initial state until the \
	computation is complete. The output is a list of snapshots of the \
	state of the QC at successive cursor-measurement times. Each \
	snapshot consists of a 4 element list whose elements are the \
	time at which the cursor is measured, the state of the register \
	immediately before the cursor is measured, the result of the \
	measurement and the state of the register immediately after \
	the cursor position is measured. The latter is the projected \
	state of the register. EvolveQC can take the optional argument \
	TimeBetweenObservations which can be set to a number or a \
	probability distribution. The default time between observations \
	is 1 time unit.";

(* Cursor is ok so long as it contains exactly 1 bit *)
CursorOkQ[{_,_,cursor_,_}]:=
	If[Count[cursor,1] == 1, True, False]

(* Computation is complete if the last cursor bit is a 1.
   The related logical test CursorOkQ, will abort the While loop
   if the cursor is ever corrupted (i.e. contains no bits or more than
   one bit). Hence the simple termination test, of checking the
   last bit, is sufficient.
*)
ComputationCompleteQ[{_, _, {___,1}, _}]:=True
ComputationCompleteQ[_]:=False

EvolnTime[time_Integer]:= time
EvolnTime[time_Real]:= time
EvolnTime[dist_[parameters___]]:=
	If[StringMatchQ[ToString[dist], "*Distribution"],
	   Random[dist[parameters]],
	   Message[EvolnTime::notknown]; Abort[]]
EvolnTime::notknown = 
	"The time between observations of the cursor must be either an \
	integer, a real number or a probability distribution.";

	
EvolveForTimeTau[{time_, 
                  stateBeforeCursorObsvd_, 
                  cursorNotAtKPlus1_, 
                  stateAfterCursorObsvd_
                 }, m_, k_, circuit_, tau_]:=
	Module[{new},
		new = SchrodingerEvolution[stateAfterCursorObsvd,m,k,circuit,tau];
		Join[{time+tau, new}, ReadCursorBits[k+1,new] ]
	]

(* Explain result creates a narrative that explains the output
   from EvolveQC
*)
ExplainResults[results_]:=
	(SetOptions[$Output, PageWidth->200];
	 Scan[ExplainResult, results])
	
ExplainResult[{t_, stateIn_, cursorFoundAt_, stateOut_}]:=
	(Print["Time t=", t];
	 Print["State of QC = ", stateIn];
	 Print["Cursor observed at position = ", cursorFoundAt];
	 Print["Collapsed state of QC = ", stateOut];
	 Print["\n"]
	)
	
(*
:[font = message; inactive; preserveAspect]
General::spell1: 
   Possible spelling error: new symbol name "explain"
     is similar to existing symbol "Explain".
:[font = message; inactive; preserveAspect; endGroup; endGroup]
General::spell1: 
   Possible spelling error: new symbol name "ExplainResults"
     is similar to existing symbol "ExplainResult".
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; endGroup; endGroup]
EvolveQC[ket[1,0,0,0], sqrtNOTcircuit]
:[font = subsection; inactive; preserveAspect; startGroup]
Plotting the Evolution
:[font = text; inactive; preserveAspect]
You can visualize the time evolution of a Feynman quantum computer using the function PlotEvolution.  PlotEvolution shows the probabilities of obtaining each possible state for the register at the cursor observation times, until the computation is done.

To use PlotEvolution you must first generate a particular evolution using EvolveQC.
:[font = input; preserveAspect; startGroup]
?PlotEvolution
:[font = print; inactive; preserveAspect; endGroup]
PlotEvolution[evolution] draws a graphic that illustrates the time evolution of
   the memory register of a Feynman quantum computer. It takes a single
   argument, the output from EvolveQC, and plots the probability (i.e.
   |amplitude|^2) of obtaining each eigenstate of the memory register at the
   times when the cursor is observed. By default PlotEvolution only plots the
   probabilities that prevail immediately before the cursor is measured.  You
   can give the option AfterObservingCursor->True to see the effect of measuring
   the cursor on the relative probabilities of finding the memory register in
   each of its possible states.
:[font = text; inactive; preserveAspect]
PlotEvolution takes a single input, the output of the function EvolveQC, and returns a graphic that shows the probability (i.e. | amplitude |^2) of each eigenstate of the memory register at the times when the cursor is observed. For compactness of notation we label the eigenstates of the (in this case 4-bit) memory register, |i>,  in base 10 notation.   For example, |5> corresponds to the eigenstate of the memory register that is really |0101> and |15> corresponds to the eigenstate of the memory register that is really |1111>. The vertical axis shows that probability of obtaining that eigenstate if the memory register were to be measured at the given time. Notice that there is a zero probability of ever obtaining certain eigenstates showing that certain configurations of the memory register are forbidden.

:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
We seed the pseudo-random number generator just to ensure reproducible results.
:[font = input; preserveAspect]
SeedRandom[1234];
evoln = EvolveQC[ket[1,0,0,0], sqrtNOTcircuit];
PlotEvolution[evoln]
:[font = text; inactive; preserveAspect; endGroup]

PlotEvolution can take two options that controls the information that is output. By default, PlotEvolution plots the probabilities of finding the computer in the various eigenstates at times t1, t2, t3 etc before the cursor position is observed. By setting the option AfterObservingCursor->True you can plot the probability of finding the computer in the various eigenstates both before and after the cursor position is observed. Hence you can visualize the effect of the cursor measurement operations by setting the option AfterObservingCursor->True.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect]
SeedRandom[1234];
evoln = EvolveQC[ket[1,0,0,0], sqrtNOTcircuit];
PlotEvolution[evoln, AfterObservingCursor->True]
:[font = text; inactive; preserveAspect; endGroup]
The time intervals between measurements need not be regular.  Try setting the time between observations, in EvolveQC, to a probability distribution using the option TimeBetweenObservations.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; endGroup]
SeedRandom[1234];
evoln = EvolveQC[ket[1,0,0,0], sqrtNOTcircuit, 
                 TimeBetweenObservations->NormalDistribution[1,0.7]];
PlotEvolution[evoln, AfterObservingCursor->True]
:[font = subsubsection; inactive; Cclosed; preserveAspect; startGroup]
Code
:[font = input; initialization; preserveAspect; endGroup; endGroup; endGroup]
*)
Options[PlotEvolution] = {BeforeObservingCursor->True, 
                          AfterObservingCursor->False};

PlotEvolution[evolnData_, opts___]:=
	Module[{opt1,opt2,times, probsB, probsA, triplesB,triplesA, graphicToPlot},
		opt1  = BeforeObservingCursor /. {opts} /. Options[PlotEvolution];
		opt2  = AfterObservingCursor  /. {opts} /. Options[PlotEvolution];
		times = Map[#[[1]]&, evolnData];
		probsB = Map[Probabilities[ #[[2]] ]&, evolnData];
		probsA = Map[Probabilities[ #[[4]] ]&, evolnData];
		
		triplesB = MapThread[BuildTriple[#1, #2]&, {times, probsB}];
		triplesA = MapThread[BuildTriple[#1, #2]&, {times, probsA}];
		
		graphicToPlot =
			Which[opt1===True && opt2===False,
					{Graphics3D[{Thickness[0.01], GrayLevel[0.5], triplesB}]},
		          opt1===True && opt2===True,
		          	{Graphics3D[{Thickness[0.01], GrayLevel[0.5], triplesB}],
		          	 Graphics3D[{Thickness[0.01], RGBColor[1,0,0], triplesA}]},
		          opt1===False && opt2===True,
		          	{Graphics3D[{Thickness[0.01], RGBColor[1,0,0], triplesA}]},
		          opt1===False && opt2===False,
		      	     Message[PlotEvolution::plottingOff]
		         ];
		Show[graphicToPlot,
			 PlotRange->{0,1},
	         Axes->True,
	         BoxRatios->{4,3,2},
	         Boxed->False,
	         AxesEdge->{{-1,-1},{1,-1},{1,1}},
	         AxesLabel->{"time", "|i>", "Pr(|i>)"},
	         FaceGrids->{{0,0,-1}, {0,1,0}, {-1,0,0}}]
	]

PlotEvolution::usage =
"PlotEvolution[evolution] draws a graphic that illustrates the \
time evolution of the memory register of a Feynman quantum \
computer. It takes a single argument, the output from EvolveQC, \
and plots the probability (i.e. |amplitude|^2) of obtaining each \
eigenstate of the memory register at the times when the cursor \
is observed. By default PlotEvolution only plots the probabilities \
that prevail immediately before the cursor is measured.  You can \
give the option AfterObservingCursor->True to see the effect of \
measuring the cursor on the relative probabilities of finding the \
memory register in each of its possible states.";

PlotEvolution::plottingOff =
	"You have turned off both plotting options so no graphics output \
	will be produced. You should set BeforeObservingCursor->True or \
	AfterObservingCursor->True or both.";

BuildTriple[time_, prob_]:=
	Line[MapIndexed[MakeStep[Flatten[{time, #2-{1}, #1}], 0.75]&, prob]]
	
MakeStep[{time_, center_, height_}, width_]:=
	Sequence[{time, center-width/2., 0},
	         {time, center-width/2., height},
	         {time, center+width/2., height},
	         {time, center+width/2., 0}
	]

(*
:[font = section; inactive; preserveAspect; startGroup]
Extracting an Answer
:[font = text; inactive; preserveAspect]
The answer is extracted by reading just the program qubits of the final state of the memory register.  To figure out which qubits are program qubits, we need to know the number of lines in the circuit (i.e. m) and the number of gates (i.e. k).  The last m qubits in the register are the program qubits and the first k+1 qubits in the register are the cursor qubits. For the sqrtNOTcircuit, m=1 and k=2.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
The state ket[1,0,0,0] represents an input of 0 to the square root of NOT squared circuit (the 4th bit is a 0, the first 3 bits are cursor bits with the cursor initialized at its start position). So we expect to see the answer 1 because the SqtNOT(SqrtNOT(0)) = NOT(0) = 1.
:[font = input; preserveAspect; endGroup]
        input = ket[0];
       cursor = ket[1,0,0];
 initialState = Direct[cursor, input];
            m = 1; 
            k = 2;
        evoln = EvolveQC[initialState, sqrtNOTcircuit];
       output = ExtractAnswer[m,k,evoln]
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
Conversely, if we input a 1 we expect to see an output of 0.
:[font = input; preserveAspect; endGroup]
        input = ket[1];
       cursor = ket[1,0,0];
 initialState = Direct[cursor, input];
            m = 1; 
            k = 2;
        evoln = EvolveQC[initialState, sqrtNOTcircuit];
       output = ExtractAnswer[m,k,evoln]
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
If we input a 0 and a 1 simultaneously the output ought to be equally likely to be 0 or 1. Try running this experiment several times.
:[font = input; preserveAspect]
        input = 1/Sqrt[2] (ket[0] + ket[1]);
       cursor = ket[1,0,0];
 initialState = Direct[cursor, Expand[input]];
            m = 1; 
            k = 2;
        evoln = EvolveQC[initialState, sqrtNOTcircuit];
       output = ExtractAnswer[m,k,evoln]
:[font = text; inactive; preserveAspect; endGroup]
The final state of the register is a superposition that weights the probability of obtaining a 0 as high as that of obtaining a 1. For example, take a particular final state and imagine measuring 20 identical copies of this state (this is just a thought experiment, you cannot copy an arbitrary quantum state exactly).
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect]
Table[ExtractAnswer[m,k,evoln], {20}]
:[font = input; preserveAspect; endGroup]
 inputBit = 1/Sqrt[2] (ket[0] + ket[1]);
 cursor = ket[1,0,0];
 initialState = Direct[cursor, inputBit]
     m = 1; 
     k = 2;
 evoln = EvolveQC[1/Sqrt[2] ket[1,0,0,0] + 1/Sqrt[2] ket[1,0,0,1], 
                  sqrtNOTcircuit];
output = ExtractAnswer[m,k,evoln]
:[font = subsubsection; inactive; Cclosed; preserveAspect; startGroup]
Code
:[font = input; initialization; preserveAspect]
*)
ExtractAnswer[m_, k_, evolution_]:=
	Module[{bitsRead},
		bitsRead = 
			ReadPartOfMemoryRegister[Last[evolution][[4]], 
	                                 Table[i, {i, k+2, m+k+1}]][[1]];
	    Apply[StringJoin, Map[ToString, bitsRead]]
	]
(*
:[font = input; initialization; preserveAspect; endGroup]
*)
(*==========================*)
(* ReadPartOfMemoryRegister *)
(*==========================*)
(* Given a superposition representing the state of the memory of a
   quantum computer, return the result of measuring a specific
   subset of the qubits in the memory.
   
   >>> This function is used in error correcting codes <<<
*)
Options[ReadPartOfMemoryRegister] = {TraceProgress->False};

ReadPartOfMemoryRegister[w_. ket[bits__] + kets_., bitsToRead_, opts___]:=
	Module[{nBits, superposition, resultsPerStep, traceQ},
		traceQ = TraceProgress /. {opts} /. Options[ReadPartOfMemoryRegister];
		nBits = Length[{bits}]; (* figure out number of bits in memory *)
		superposition = {"BeforeAnyMeasurements", w ket[bits] + kets};
		resultsPerStep = FoldList[MeasureIthBit[#2,#1,nBits]&,
		                          superposition,
		                          bitsToRead
		                         ];
		Which[traceQ===False, 
		        {Rest[Map[#[[1]]&, resultsPerStep]],  (* results for bits *)
		         Last[resultsPerStep][[2]]            (* projected state *)
		        },
		      traceQ===True,
		        ColumnForm[resultsPerStep] (*list of {results,projectedStates}*)
		     ]
	]
	
ReadPartOfMemoryRegister::usage =
	"ReadPartOfMemoryRegister[superposition, bitsToRead] reads the state \
	of selected bits in the memory register. As the i-th and j-th bit \
	measurement operators commute (for any i and j), it does not matter \
	in what order you measure the bits.";
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]

:[font = input; initialization; preserveAspect; endGroup; endGroup]
*)
On[General::spell1]
(*
:[font = section; inactive; Cclosed; preserveAspect; startGroup]
Bras and Kets defined in here ...
:[font = subsection; inactive; preserveAspect; startGroup]
Converting Kets to Column Vectors
:[font = input; initialization; preserveAspect; endGroup]
*)
KetToColumnVector[ket[0]]:={{1},{0}}      (* spin up   = 0 *)
KetToColumnVector[ket[1]]:={{0},{1}}      (* spin down = 1 *)
KetToColumnVector[ket[bits__]]:=
	Apply[Direct, Map[KetToColumnVector[ket[#]]&, {bits}]]

KetToColumnVector[a_ ket_ket]:=
	a KetToColumnVector[ket]
	
KetToColumnVector[Plus[ket_, kets___]]:=
	Apply[Plus, Map[KetToColumnVector, {ket, kets}]]
	
KetToColumnVector[superposition_]:=
	KetToColumnVector[ Expand[superposition] ] 
(*
:[font = subsection; inactive; preserveAspect; startGroup]
Converting Bras to Row Vectors
:[font = input; initialization; preserveAspect; endGroup]
*)
BraToRowVector[bra[0]]:={{1,0}}
BraToRowVector[bra[1]]:={{0,1}}
BraToRowVector[w_. bra[bits__]]:=
	w * Apply[Direct, Map[BraToRowVector[bra[#]]&, {bits}]]
BraToRowVector[w_. bra[bits__] + bras_.]:=
	BraToRowVector[w * bra[bits]] + BraToRowVector[bras]
BraToRowVector[superposition_]:=
	BraToRowVector[Expand[superposition]]
(*
:[font = subsection; inactive; preserveAspect; startGroup]
Converting Column Vectors to Kets
:[font = input; initialization; preserveAspect; endGroup]
*)
ColumnVectorToKet[amplitudes_]:=
	Apply[Plus,
		  MapThread[(#1[[1]] #2)&,
		            {amplitudes,
		             EigenKets[ Length[amplitudes] ]
		            }
		           ]
		 ]
(*
:[font = subsection; inactive; preserveAspect; startGroup]
Converting Row Vectors To Bras
:[font = input; initialization; preserveAspect; endGroup]
*)
RowVectorToBra[{{wi__}}]:=
	Module[{eigenBras},
		eigenBras = EigenKets[Length[{wi}]] /. ket->bra;
		Apply[Plus, MapThread[(#1 #2)&, {{wi}, eigenBras}]]
	]
(*
:[font = subsection; inactive; preserveAspect; startGroup]
Converting Between Bras and Kets
:[font = input; initialization; preserveAspect; endGroup]
*)
KetToBra[ket_]:=
	RowVectorToBra[Conjugate[Transpose[KetToColumnVector[ket]]]]

BraToKet[bra_]:=
	ColumnVectorToKet[Conjugate[Transpose[BraToRowVector[bra]]]]
(*
:[font = subsection; inactive; preserveAspect; startGroup]
Average Value of an Observable
:[font = input; initialization; preserveAspect; endGroup]
*)
ExpectationValue[w_. ket[bits__] + kets_., observable_]:=
	(If[!HermitianQ[observable], 
		(Message[ExpectationValue::notHermitian]; Abort[]),
		If[Length[observable] != 2^Length[{bits}],
		   (Message[ExpectationValue::incompatible]; Abort[])]];
		       
	 (BraToRowVector[KetToBra[w * ket[bits] + kets]] . 
	  observable . 
	  KetToColumnVector[w * ket[bits] + kets]
	 )[[1,1]]  (* scalar = a 1 x 1 matrix, [[1,1]] removes the parentheses *)
	)

ExpectationValue[superposition_, observable_]:=
	ExpectationValue[Expand[superposition], observable]

ExpectationValue::notHermitian =
	"Your purported observable is not an Hermitian matrix.";
ExpectationValue::incompatible =
	"The dimensions of the state vector and observable are incompatible.";

(*
:[font = subsection; inactive; preserveAspect; startGroup]
Creating Eigenstates that Span a Hilbert Space
:[font = input; initialization; preserveAspect]
*)
BasisEigenstates[m_Integer]:= EigenKets[2^m]

BasisEigenstates::usage = 
  "BasisEigenstates[m] returns the complete set of \
  eigenstates that span the Hilbert space of an m-bit \
  quantum memory register.";
(*
:[font = input; initialization; preserveAspect; endGroup]
*)
EigenKets[n_]:=
	Module[{bits},
		bits = Table[Apply[ket, IntegerDigits[i,2]], 
		             {i, 0, n-1}];
		          (* last eigenket has the most bits *)
		Map[PadTo[Length[Last[bits]], #]&, bits]
	]

PadTo[nDigits_, digits_]:=
	Join[Apply[ket, Table[0,{nDigits - Length[digits]}]], 
	     digits]
(*
:[font = subsection; inactive; preserveAspect; startGroup]
Accessing Amplitudes of Superpositions and Computing Probabilities
:[font = input; initialization; preserveAspect]
*)
Options[Amplitudes] = {ShowEigenstates->False};

ShowEigenstates::usage = 
	"ShowEigenstates is an option for Amplitudes that \
	determines whether the 
output should be a list of the \
	amplitudes or a list of {eigenstate, 
amplitude} pairs.";

Amplitudes[w_. ket[bits__] + kets_., opts___]:=
	Module[{showeigen},
	showeigen = ShowEigenstates /. {opts} /. Options[Amplitudes];
	Which[showeigen == True, 
			Map[{#, Coefficient[w ket[bits] + kets, #]}&,
		        BasisEigenstates[ Length[{bits}] ]
		       ],
		  showeigen == False,
		    Map[Coefficient[w ket[bits] + kets, #]&,
		        BasisEigenstates[ Length[{bits}] ]
		       ]
		  ]
	]

(* This clause catches cases like 1/Sqrt[2] (ket[0] + ket[1]) etc *)	
Amplitudes[c_ (w_. ket[bits__] + kets_.)]:=
	Amplitudes[ Expand[c (w ket[bits] + kets)] ]

Amplitudes::usage = 
  "Amplitudes[superposition] returns the amplitudes of the \
  eigenstates in a superposition or ket vectors.";
(*
:[font = input; initialization; preserveAspect]
*)
Options[Probabilities] = {ShowEigenstates->False};

Probabilities[w_. ket[bits__] + kets_., opts___]:=
	Module[{showeigen, amplitudes, symbols, sumOfSquares},
		showeigen    = ShowEigenstates /. {opts} /. Options[Probabilities];
		amplitudes   = Amplitudes[w ket[bits] + kets];
		symbols      = SymbolicCoefficients[amplitudes]; (*see below*)
		sumOfSquares = Simplify[
		                Apply[Plus, 
		                      Map[ComplexExpand[Abs[#]^2, symbols]&, 
		                          amplitudes]]];
		amplitudes   = If[sumOfSquares=!=1,  (* renormalize amplitudes
		                                         if necessary *)
		                  amplitudes/Sqrt[sumOfSquares],
		                  amplitudes];
		Which[showeigen == True,  
		       MapThread[{#1, ComplexExpand[Abs[#2]^2, symbols]}&, 
		                 {BasisEigenstates[Length[{bits}]], amplitudes}
		                ],
			  showeigen == False, 
			   Map[ComplexExpand[Abs[#]^2, symbols]&, amplitudes]
	    ]
	]

Probabilities[c_ (w_. ket[bits__] + kets_.)]:=
	Probabilities[ Expand[c (w ket[bits] + kets)] ]

Probabilities::usage =
	"Probabilities[superposition] returns the probabilities of \
	 finding a system in a state described by superposition in \
	 each of its possible eigenstates upon being measured (observed). \
	 If Probabilities is given the option ShowEigenstates->True \
	 the function returns a list of {eigenstate, probability} pairs.";
(*
:[font = input; initialization; preserveAspect; endGroup]
*)
SymbolicCoefficients[amplitudes_List]:=
	Select[Union[Flatten[Map[Variables, amplitudes]]], 
		   Not[MatchQ[#, Abs[_]]]&]
(*
:[font = subsection; inactive; preserveAspect; startGroup]
Testing Whether a Ket is Properly Normalized
:[font = input; initialization; preserveAspect; endGroup]
*)
Needs["Algebra`ReIm`"];

NormalizedKetQ[ket_]:=
	Module[{columnVector},
		columnVector = KetToColumnVector[ket];
		(Inner[Times, 
		       Conjugate[Transpose[columnVector]], 
               columnVector,
               Plus
              ] == {{1}} // N ) /. z_ Conjugate[z_] :> Abs[z]^2
    ]
   
NormalizedKetQ::usage =
	"NormalizedKetQ[ket] returns True if the square \
	moduli of the amplitudes of the eigenkets in the \
	superposition \"ket\" sum to 1. If \"ket\" has non-numeric \
	amplitudes, the normalization cannot always be determined.";
(*
:[font = subsection; inactive; preserveAspect; startGroup]
NormalizeKet
:[font = input; initialization; preserveAspect; endGroup]
*)
NormalizeKet[superposition_]:=
	superposition /; NormalizedKetQ[superposition]
NormalizeKet[superposition_]:=
	Expand[superposition / 
	       Sqrt[Apply[Plus, 
	                  Map[Abs[#]^2&, 
	                      Amplitudes[superposition, 
	                                 ShowEigenstates->False]
	                     ]
	                 ]
	           ]
	]
	      
NormalizeKet::usage =
	"NormalizeKet[superposition] is used to normalize a given \
	superposition of
 ket vectors. That is, if the sum of the squares \
	of the absolute values of 
the amplitudes of the eigenstates in \
	the superposition do not sum to 1, 
NormalizeKet rescales the \
	amplitudes so that they squared moduli will sum 
to 1.";
(*
:[font = subsection; inactive; preserveAspect; startGroup]
Direct Product
:[font = input; initialization; preserveAspect; endGroup]
*)
(* Last modified 09/07/96 *)
Needs["LinearAlgebra`MatrixManipulation`"];

Direct[op1_, op2_]:=
	BlockMatrix[Outer[Times, op1, op2]] /; MatrixQ[op1] && MatrixQ[op2]
	
Direct[ket_, bra_]:=
	Direct[KetToColumnVector[ket], BraToRowVector[bra]] /; IsKetQ[ket] && IsBraQ[
bra]
	
Direct[ket1_, ket2_]:=
	ColumnVectorToKet[
		Direct[KetToColumnVector[ket1],
	           KetToColumnVector[ket2]]
	]/; IsKetQ[ket1] && IsKetQ[ket2]

Direct[bra1_, bra2_]:=
	RowVectorToBra[
		Direct[BraToRowVector[bra1],
			   BraToRowVector[bra2]]
	] /; IsBraQ[bra1] && IsBraQ[bra2]
	
Direct[bra_, ket_]:=
	(Message[Direct::braket];
	 Direct[BraToRowVector[bra], KetToColumnVector[ket]]) /; IsBraQ[bra] && 
IsKetQ[ket]

Direct[bra_, op_]:=
	(Message[Direct::braop];
	 Direct[BraToRowVector[bra], op]) /; IsBraQ[bra] && MatrixQ[op]
	
Direct[op_, bra_]:=
	(Message[Direct::opbra];
	 Direct[op, BraToRowVector[bra]]) /; MatrixQ[op] && IsBraQ[bra]
	
Direct[ket_, op_]:=
	(Message[Direct::ketop];
	 Direct[KetToColumnVector[ket], op]) /; IsKetQ[ket] && MatrixQ[op]
	
Direct[op_, ket_]:=
	(Message[Direct::opket];
	 Direct[op, KetToColumnVector[ket]]) /; MatrixQ[op] && IsKetQ[ket]

Direct[matrices__]:=
	Fold[Direct, First[{matrices}], Rest[{matrices}]]

Direct::braket =
	"Warning - You are taking the DIRECT product of a bra \
	and a ket. This is 
unusual. Perhaps you meant to use \
	the DOT product?";
	
Direct::braop =
	"Warning - You are taking the DIRECT product of a bra \
	with an operator. 
This is unusual. Perhaps you meant to use \
	the DOT product?";
	
Direct::opbra =
	"Warning - You are taking the DIRECT product of an operator \
	with a bra. 
This is unusual. Perhaps you meant to use \
	the DOT product?";

Direct::ketop =
	"Warning - You are taking the DIRECT product of a ket \
	with an operator. 
This is unusual. Perhaps you meant to use \
	the DOT product?";

Direct::opket =
	"Warning - You are taking the DIRECT product of an operator \
	with a ket. 
This is unusual. Perhaps you meant to use \
	the DOT product?";


IsKetQ[w_. ket[__] + kets_.]:= True
IsKetQ[_]:=False
	
IsBraQ[w_. bra[__] + bras_.]:= True
IsBraQ[_]:=False
(*
:[font = subsection; inactive; preserveAspect; startGroup]
Truth Table of a Logic Gate
:[font = input; initialization; preserveAspect; endGroup]
*)
TruthTable[gate_]:=
	Module[{n,m},
		{n,m} = Dimensions[gate];
		Which[Not[n==m && IntegerQ[n] && IntegerQ[m]],
		      Message[TruthTable::notsquare]; Abort[],
		      Not[IntegerQ[Log[2, n]]],
		      Message[TruthTable::powerof2]; Abort[]
		     ];
		Map[(# -> ColumnVectorToKet[gate . KetToColumnVector[#]])&, 
		    EigenKets[n]
		   ]  // ColumnForm
	]
	
TruthTable::notsquare = 
  "Your input is not a square matrix and cannot, therefore, represent a \
  
reversible logic gate.";

TruthTable::powerof2 = 
  "Your input is not a matrix of dimensions (2^m) x (2^m) for integer m \
  
and cannot, therefore, represent a reversible logic gate that operates \
  on 
m bits.";
(*
:[font = subsection; inactive; preserveAspect; startGroup]
Types of Operators (Matrices)
:[font = input; preserveAspect]
HermitianQ[matrix_]:=
	matrix == Conjugate[Transpose[matrix]]
:[font = input; initialization; preserveAspect; endGroup]
*)
UnitaryQ[matrix_]:=
  Module[{rows, cols},
	{rows, cols} = Dimensions[matrix];
	If[Not[IntegerQ[rows]] || 
	   Not[IntegerQ[cols]] || 
	   rows != cols, Message[UnitaryQ::notsquarematrix]];
	
	   Chop[Simplify[ComplexExpand[Conjugate[Transpose[matrix]]] - 
	                 ComplexExpand[Inverse[matrix]]
	                ]
	       ] == ZeroMatrix[rows, cols]
  ]

UnitaryQ::notsquarematrix =
  "Your input is not a square matrix.";
  
ZeroMatrix[rows_, cols_]:=
	Table[0, {rows}, {cols}]
(*
:[font = subsection; inactive; preserveAspect; startGroup]
Tools for Making Test Superpositions
:[font = input; initialization; preserveAspect]
*)
SymbolicSuperposition[m_]:=
	Apply[Plus,
	      MapThread[(#1 #2)&, 
	                {SymbolicAmplitudes[m], BasisEigenstates[m]}]
	]
	
SymbolicSuperposition::usage =
	"SymbolicSuperposition[m] creates a superposition of 2^m \
	eigenstates whose
 amplitudes are uninstantiated symbols. These \
	eigenstates represent the 
possible states of an m-bit memory \
	register of a quantum computer. This 
function is useful for \
	exploring the effects of quantum mechanical 
operations on \
	arbitrary superpositions. Note that the general form does not

	guarentee that the superposition is normalized.";
	
SymbolicAmplitudes[m_]:=
	(Clear[w];
	 Map[ToExpression["w"<>ToString[#]]&, Table[i,{i,0,2^m - 1}]]
	)
(*
:[font = input; initialization; preserveAspect; endGroup; endGroup; endGroup]
*)
Options[RandomSuperposition] = {Normalized->True};

RandomSuperposition[m_, opts___]:=
	Module[{normalized},
		normalized = Normalized /. {opts} /. Options[RandomSuperposition];
		superposition = Apply[Plus,
	                          MapThread[(#1 #2)&, 
	                                    {RandomAmplitudes[m],
	                                     BasisEigenstates[m]}
	                                   ]
	                         ];
		Which[normalized==True, NormalizeKet[superposition],
	          normalized==False, superposition
	         ]
	]

RandomSuperposition::usage =
	"RandomSuperposition[m] creates a normalized superposition \
	of 2^m eigenstates whose amplitudes are random complex numbers. \
	These eigenstates represent the possible states of an m-bit \
	memory register of a quantum computer. You can generate an \
	un-normalized superposition by setting the option Normalized->False.";
	
(* You can pick the amplitudes according to whatever distribution
   you like. In the current case we pick random complex numbers
   uniformly from the square in the complex plane bounded by a lower
   left corner at (-1,-I) and an upper right corner at (1,I).
*)
RandomAmplitudes[m_]:=
	Table[Random[Complex, {-1-I, 1+I}], {2^m}]
(*
^*)
